<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;
use App\Models\Inquiry;

class Quotation extends Model
{
    use HasFactory;

    protected $fillable = [
        'inquiry_id',
        'quotation_number',
        'costing_details',
        'calculated_total_fabric_consumption',
        'calculated_total_fabric_cost',
        'calculated_accessories_cost',
        'calculated_additional_costs_total',
        'calculated_total_cost',
        'quoted_price',
        'remarks',
        'status',
        'status_updated_at',
        'status_updated_by',
        'status_notes',
    ];

    protected $casts = [
        'costing_details' => 'array',
        'calculated_total_fabric_consumption' => 'decimal:4',
        'calculated_total_fabric_cost' => 'decimal:2',
        'calculated_accessories_cost' => 'decimal:2',
        'calculated_additional_costs_total' => 'decimal:2',
        'calculated_total_cost' => 'decimal:2',
        'quoted_price' => 'decimal:2',
        'status_updated_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Boot the model.
     */
    protected static function boot()
    {
        parent::boot();

        // Track status changes
        static::updating(function ($quotation) {
            if ($quotation->isDirty('status')) {
                $quotation->status_updated_at = now();
                $quotation->status_updated_by = auth()->id();
            }
        });
    }

    /**
     * Get the inquiry associated with the quotation.
     */
    public function inquiry(): BelongsTo
    {
        return $this->belongsTo(Inquiry::class);
    }

    /**
     * Get the order confirmation associated with the quotation.
     */
    public function orderConfirmation(): HasOne
    {
        return $this->hasOne(OrderConfirmation::class);
    }

    /**
     * Get the user who last updated the status.
     */
    public function statusUpdatedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'status_updated_by');
    }

    /**
     * Scope a query to only include quotations with a specific status.
     */
    public function scopeWithStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope a query to only include pending quotations.
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope a query to only include won quotations.
     */
    public function scopeWon($query)
    {
        return $query->where('status', 'won');
    }

    /**
     * Scope a query to only include lost quotations.
     */
    public function scopeLost($query)
    {
        return $query->where('status', 'lost');
    }

    /**
     * Get the status badge color for UI display.
     */
    public function getStatusBadgeColorAttribute(): string
    {
        return match($this->status) {
            'draft' => 'secondary',
            'submitted' => 'warning',
            'approved' => 'info',
            'rejected' => 'danger',
            'pending' => 'primary',
            'won' => 'success',
            'lost' => 'dark',
            default => 'secondary',
        };
    }

    /**
     * Get the status label for UI display.
     */
    public function getStatusLabelAttribute(): string
    {
        return match($this->status) {
            'draft' => 'Draft',
            'submitted' => 'Submitted',
            'approved' => 'Approved',
            'rejected' => 'Rejected',
            'pending' => 'Pending',
            'won' => 'Won',
            'lost' => 'Lost',
            default => ucfirst($this->status),
        };
    }

    /**
     * Check if quotation can be converted to order.
     */
    public function getCanConvertToOrderAttribute(): bool
    {
        return $this->status === 'won' && !$this->orderConfirmation;
    }

    /**
     * Check if quotation has an associated order.
     */
    public function getHasOrderAttribute(): bool
    {
        return $this->orderConfirmation !== null;
    }

    /**
     * Get the profit margin.
     */
    public function getProfitMarginAttribute(): float
    {
        if ($this->calculated_total_cost == 0) {
            return 0;
        }
        
        return (($this->quoted_price - $this->calculated_total_cost) / $this->calculated_total_cost) * 100;
    }

    /**
     * Get the profit amount.
     */
    public function getProfitAmountAttribute(): float
    {
        return $this->quoted_price - $this->calculated_total_cost;
    }

    /**
     * Update quotation status with notes.
     */
    public function updateStatus(string $status, string $notes = null): bool
    {
        $this->status = $status;
        $this->status_notes = $notes;
        $this->status_updated_at = now();
        $this->status_updated_by = auth()->id();
        
        return $this->save();
    }

    /**
     * Convert quotation to order confirmation.
     */
    public function convertToOrder(array $orderData = []): OrderConfirmation
    {
        if (!$this->can_convert_to_order) {
            throw new \Exception('Quotation cannot be converted to order');
        }

        $orderData = array_merge([
            'quotation_id' => $this->id,
            'confirmed_quantity' => $this->inquiry->order_quantity,
            'confirmed_price' => $this->quoted_price,
            'created_by' => auth()->id(),
        ], $orderData);

        return OrderConfirmation::create($orderData);
    }

    /**
     * Get available status options.
     */
    public static function getStatusOptions(): array
    {
        return [
            'draft' => 'Draft',
            'submitted' => 'Submitted',
            'approved' => 'Approved',
            'rejected' => 'Rejected',
            'pending' => 'Pending',
            'won' => 'Won',
            'lost' => 'Lost',
        ];
    }
}

