<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class PaymentAllocation extends Model
{
    use HasFactory;

    protected $fillable = [
        'advance_payment_id',
        'factory_id',
        'order_confirmation_id',
        'amount',
        'allocation_date',
        'purpose',
        'reference_number',
        'remarks',
        'created_by'
    ];

    protected $casts = [
        'allocation_date' => 'date',
        'amount' => 'decimal:2',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($allocation) {
            if (empty($allocation->reference_number)) {
                $allocation->reference_number = static::generateReferenceNumber();
            }
        });

        static::created(function ($allocation) {
            $allocation->advancePayment->increment('allocated_amount', $allocation->amount);
            $allocation->advancePayment->decrement('remaining_amount', $allocation->amount);
        });

        static::deleted(function ($allocation) {
            $allocation->advancePayment->decrement('allocated_amount', $allocation->amount);
            $allocation->advancePayment->increment('remaining_amount', $allocation->amount);
        });
    }

    public static function generateReferenceNumber(): string
    {
        $lastAllocation = static::latest('id')->first();
        $nextNumber = $lastAllocation ? $lastAllocation->id + 1 : 1;
        return 'ALLOC-' . date('Y') . '-' . str_pad($nextNumber, 5, '0', STR_PAD_LEFT);
    }

    public function advancePayment(): BelongsTo
    {
        return $this->belongsTo(AdvancePayment::class);
    }

    public function factory(): BelongsTo
    {
        return $this->belongsTo(Factory::class);
    }

    public function orderConfirmation(): BelongsTo
    {
        return $this->belongsTo(OrderConfirmation::class);
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }
}