<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class OrderStatusHistory extends Model
{
    use HasFactory;
    protected $table = 'order_status_history'; // exact table name in DB

    protected $fillable = [
        'order_confirmation_id',
        'previous_status',
        'new_status',
        'notes',
        'changed_by',
        'changed_at',
    ];

    protected $casts = [
        'changed_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * The attributes that should be hidden for serialization.
     */
    protected $hidden = [];

    /**
     * Get the order confirmation that owns the status history.
     */
    public function orderConfirmation(): BelongsTo
    {
        return $this->belongsTo(OrderConfirmation::class);
    }

    /**
     * Get the user who changed the status.
     */
    public function changedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'changed_by');
    }

    /**
     * Scope a query to order by most recent changes first.
     */
    public function scopeLatest($query)
    {
        return $query->orderBy('changed_at', 'desc');
    }

    /**
     * Get the formatted change date.
     */
    public function getFormattedChangeDateAttribute(): string
    {
        return $this->changed_at->format('M d, Y H:i');
    }

    /**
     * Get the status change description.
     */
    public function getChangeDescriptionAttribute(): string
    {
        $from = $this->previous_status ? ucfirst(str_replace('_', ' ', $this->previous_status)) : 'New';
        $to = ucfirst(str_replace('_', ' ', $this->new_status));
        
        return "Status changed from {$from} to {$to}";
    }

    /**
     * Get the badge color for the new status.
     */
    public function getStatusBadgeColorAttribute(): string
    {
        return match($this->new_status) {
            'pending' => 'secondary',
            'confirmed' => 'primary',
            'in_production' => 'warning',
            'shipped' => 'info',
            'delivered' => 'success',
            'cancelled' => 'danger',
            default => 'secondary',
        };
    }

    /**
     * Get the icon for the status change.
     */
    public function getStatusIconAttribute(): string
    {
        return match($this->new_status) {
            'pending' => 'fas fa-clock',
            'confirmed' => 'fas fa-check-circle',
            'in_production' => 'fas fa-cogs',
            'shipped' => 'fas fa-shipping-fast',
            'delivered' => 'fas fa-check-double',
            'cancelled' => 'fas fa-times-circle',
            default => 'fas fa-info-circle',
        };
    }
}

