<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Storage;

class OrderDocument extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_confirmation_id',
        'document_type',
        'document_name',
        'file_path',
        'file_size',
        'mime_type',
        'description',
        'uploaded_by',
        'uploaded_at',
    ];

    protected $casts = [
        'uploaded_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * The attributes that should be hidden for serialization.
     */
    protected $hidden = [];

    /**
     * Boot the model.
     */
    protected static function boot()
    {
        parent::boot();

        // Delete file when document is deleted
        static::deleting(function ($document) {
            if (Storage::exists($document->file_path)) {
                Storage::delete($document->file_path);
            }
        });
    }

    /**
     * Get the order confirmation that owns the document.
     */
    public function orderConfirmation(): BelongsTo
    {
        return $this->belongsTo(OrderConfirmation::class);
    }

    /**
     * Get the user who uploaded the document.
     */
    public function uploader(): BelongsTo
    {
        return $this->belongsTo(User::class, 'uploaded_by');
    }

    /**
     * Scope a query to only include documents of a specific type.
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('document_type', $type);
    }

    /**
     * Get the document type label for UI display.
     */
    public function getDocumentTypeLabelAttribute(): string
    {
        return match($this->document_type) {
            'purchase_order' => 'Purchase Order',
            'invoice' => 'Invoice',
            'packing_list' => 'Packing List',
            'shipping_document' => 'Shipping Document',
            'quality_certificate' => 'Quality Certificate',
            'fabric_specification' => 'Fabric Specification',
            'production_schedule' => 'Production Schedule',
            'other' => 'Other Document',
            default => ucfirst(str_replace('_', ' ', $this->document_type)),
        };
    }

    /**
     * Get the file size in human readable format.
     */
    public function getFormattedFileSizeAttribute(): string
    {
        if (!$this->file_size) {
            return 'Unknown';
        }

        $bytes = $this->file_size;
        $units = ['B', 'KB', 'MB', 'GB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }

    /**
     * Get the file extension.
     */
    public function getFileExtensionAttribute(): string
    {
        return pathinfo($this->file_path, PATHINFO_EXTENSION);
    }

    /**
     * Check if the document is an image.
     */
    public function getIsImageAttribute(): bool
    {
        $imageTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        return in_array($this->mime_type, $imageTypes);
    }

    /**
     * Check if the document is a PDF.
     */
    public function getIsPdfAttribute(): bool
    {
        return $this->mime_type === 'application/pdf';
    }

    /**
     * Get the download URL for the document.
     */
    public function getDownloadUrlAttribute(): string
    {
        return route('admin.order-confirmations.download-document', [
            'order' => $this->order_confirmation_id,
            'document' => $this->id
        ]);
    }

    /**
     * Get the icon class for the document type.
     */
    public function getIconClassAttribute(): string
    {
        if ($this->is_image) {
            return 'fas fa-image text-info';
        }
        
        if ($this->is_pdf) {
            return 'fas fa-file-pdf text-danger';
        }
        
        return match($this->file_extension) {
            'doc', 'docx' => 'fas fa-file-word text-primary',
            'xls', 'xlsx' => 'fas fa-file-excel text-success',
            'zip', 'rar' => 'fas fa-file-archive text-warning',
            default => 'fas fa-file text-secondary',
        };
    }

    /**
     * Get available document types.
     */
    public static function getDocumentTypes(): array
    {
        return [
            'purchase_order' => 'Purchase Order',
            'invoice' => 'Invoice',
            'packing_list' => 'Packing List',
            'shipping_document' => 'Shipping Document',
            'quality_certificate' => 'Quality Certificate',
            'fabric_specification' => 'Fabric Specification',
            'production_schedule' => 'Production Schedule',
            'other' => 'Other Document',
        ];
    }
}

