<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class OrderConfirmation extends Model
{
    use HasFactory;

    protected $fillable = [
        'quotation_id',
        'order_number',
        'customer_po_number',
        'confirmed_quantity',
        'confirmed_price',
        'delivery_date',
        'payment_terms',
        'special_instructions',
        'pi_file',
        'status',
        'created_by',
        'updated_by',
    ];

    protected $casts = [
        'confirmed_price' => 'decimal:2',
        'delivery_date' => 'date',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * The attributes that should be hidden for serialization.
     */
    protected $hidden = [];

    /**
     * Boot the model.
     */
    protected static function boot()
    {
        parent::boot();

        // Auto-generate order number when creating
        static::creating(function ($orderConfirmation) {
            if (empty($orderConfirmation->order_number)) {
                $orderConfirmation->order_number = static::generateOrderNumber();
            }
        });

        // Track status changes
        // static::updating(function ($orderConfirmation) {
        //     if ($orderConfirmation->isDirty('status')) {
        //         $orderConfirmation->recordStatusChange();
        //     }
        // });
    }

    /**
     * Generate unique order number
     */
    public static function generateOrderNumber(): string
    {
        $lastOrder = static::latest('id')->first();
        $nextNumber = $lastOrder ? $lastOrder->id + 1 : 1;
        return 'ORD-' . date('Y') . '-' . str_pad($nextNumber, 5, '0', STR_PAD_LEFT);
    }

    /**
     * Record status change in history
     */
    // protected function recordStatusChange(): void
    // {
    //     if ($this->isDirty('status')) {
    //         $this->statusHistory()->create([
    //             'previous_status' => $this->getOriginal('status'),
    //             'new_status' => $this->status,
    //             'changed_by' => auth()->id(),
    //             'changed_at' => now(),
    //         ]);
    //     }
    // }

    /**
     * Get the quotation that owns the order confirmation.
     */
    public function quotation(): BelongsTo
    {
        return $this->belongsTo(Quotation::class);
    }

    /**
     * Get the documents for the order confirmation.
     */
    public function documents(): HasMany
    {
        return $this->hasMany(OrderDocument::class);
    }

    /**
     * Get the status history for the order confirmation.
     */
    public function statusHistory(): HasMany
    {
        return $this->hasMany(OrderStatusHistory::class);
    }

    /**
     * Get the user who created the order confirmation.
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the user who last updated the order confirmation.
     */
    public function updater(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * Scope a query to only include orders with a specific status.
     */
    public function scopeWithStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope a query to only include orders due for delivery.
     */
    public function scopeDueForDelivery($query, $days = 7)
    {
        return $query->where('delivery_date', '<=', now()->addDays($days))
                    ->whereIn('status', ['confirmed', 'in_production']);
    }

    /**
     * Get the status badge color for UI display.
     */
    public function getStatusBadgeColorAttribute(): string
    {
        return match($this->status) {
            'pending' => 'secondary',
            'confirmed' => 'primary',
            'in_production' => 'warning',
            'shipped' => 'info',
            'delivered' => 'success',
            'cancelled' => 'danger',
            default => 'secondary',
        };
    }

    /**
     * Get the formatted delivery date.
     */
    public function getFormattedDeliveryDateAttribute(): string
    {
        return $this->delivery_date ? $this->delivery_date->format('M d, Y') : 'Not set';
    }

    /**
     * Get the total value of the order.
     */
    public function getTotalValueAttribute(): float
    {
        return $this->confirmed_quantity * $this->confirmed_price;
    }

    /**
     * Check if order is overdue.
     */
    public function getIsOverdueAttribute(): bool
    {
        return $this->delivery_date && 
               $this->delivery_date->isPast() && 
               !in_array($this->status, ['delivered', 'cancelled']);
    }
public function seller() {
    return $this->belongsTo(Seller::class);
}
public function term() {
    return $this->belongsTo(Term::class);
}

    /**
     * Get days until delivery.
     */
    public function getDaysUntilDeliveryAttribute(): ?int
    {
        if (!$this->delivery_date) {
            return null;
        }
        
        return now()->diffInDays($this->delivery_date, false);
    }
    
}

