<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class BookingInvoice extends Model
{
    use HasFactory;

    protected $fillable = [
        'shipment_id',
        'invoice_number',
        'invoice_date',
        'freight_charges',
        'insurance_charges',
        'handling_charges',
        'other_charges',
        'total_charges',
        'charges_description',
        'document_path',
        'created_by'
    ];

    protected $casts = [
        'invoice_date' => 'date',
        'freight_charges' => 'decimal:2',
        'insurance_charges' => 'decimal:2',
        'handling_charges' => 'decimal:2',
        'other_charges' => 'decimal:2',
        'total_charges' => 'decimal:2',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($invoice) {
            if (empty($invoice->invoice_number)) {
                $invoice->invoice_number = static::generateInvoiceNumber();
            }
            // Calculate total charges
            $invoice->total_charges = $invoice->freight_charges + 
                                    $invoice->insurance_charges + 
                                    $invoice->handling_charges + 
                                    $invoice->other_charges;
        });

        static::updating(function ($invoice) {
            $invoice->total_charges = $invoice->freight_charges + 
                                    $invoice->insurance_charges + 
                                    $invoice->handling_charges + 
                                    $invoice->other_charges;
        });
    }

    public static function generateInvoiceNumber(): string
    {
        $lastInvoice = static::latest('id')->first();
        $nextNumber = $lastInvoice ? $lastInvoice->id + 1 : 1;
        return 'BINV-' . date('Y') . '-' . str_pad($nextNumber, 5, '0', STR_PAD_LEFT);
    }

    public function shipment(): BelongsTo
    {
        return $this->belongsTo(Shipment::class);
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }
}