<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class AdvancePayment extends Model
{
    use HasFactory;

    protected $fillable = [
        'buyer_id',
        'order_confirmation_id',
        'reference_number',
        'amount',
        'payment_date',
        'payment_method',
        'transaction_id',
        'remarks',
        'status',
        'allocated_amount',
        'remaining_amount',
        'created_by'
    ];

    protected $casts = [
        'payment_date' => 'date',
        'amount' => 'decimal:2',
        'allocated_amount' => 'decimal:2',
        'remaining_amount' => 'decimal:2',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($advancePayment) {
            if (empty($advancePayment->reference_number)) {
                $advancePayment->reference_number = static::generateReferenceNumber();
            }
            $advancePayment->remaining_amount = $advancePayment->amount;
        });

        static::created(function ($advancePayment) {
            $advancePayment->updateLedger();
        });

        static::updated(function ($advancePayment) {
            $advancePayment->updateLedger();
        });
    }

    public static function generateReferenceNumber(): string
    {
        $lastPayment = static::latest('id')->first();
        $nextNumber = $lastPayment ? $lastPayment->id + 1 : 1;
        return 'ADV-' . date('Y') . '-' . str_pad($nextNumber, 5, '0', STR_PAD_LEFT);
    }

    public function updateLedger(): void
    {
        // Update ledger balances
        $ledger = PaymentLedger::firstOrCreate(['buyer_id' => $this->buyer_id]);
        $ledger->updateBalance();
    }

    public function buyer(): BelongsTo
    {
        return $this->belongsTo(Buyer::class);
    }

    public function orderConfirmation(): BelongsTo
    {
        return $this->belongsTo(OrderConfirmation::class);
    }

    public function allocations(): HasMany
    {
        return $this->hasMany(PaymentAllocation::class);
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function getStatusBadgeColorAttribute(): string
    {
        return match($this->status) {
            'pending' => 'warning',
            'received' => 'success',
            'cancelled' => 'danger',
            default => 'secondary',
        };
    }

    public function canAllocate(): bool
    {
        return $this->status === 'received' && $this->remaining_amount > 0;
    }
}