<?php

namespace App\Http\Controllers;

use App\Models\Shipment;
use App\Models\OrderConfirmation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ShipmentController extends Controller
{
    public function index()
    {
        $shipments = Shipment::with(['orderConfirmation', 'creator'])
            ->latest()
            ->paginate(20);

        return view('admin.shipment.shipments.index', compact('shipments'));
    }

    public function create()
    {
        $orders = OrderConfirmation::whereIn('status', ['confirmed', 'in_production', 'shipped'])
            ->get();
            
        return view('admin.shipment.shipments.create', compact('orders'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'order_confirmation_id' => 'required|exists:order_confirmations,id',
            'shipment_date' => 'required|date',
            'vessel_flight_no' => 'nullable|string|max:100',
            'bill_of_lading' => 'nullable|string|max:100',
            'port_of_loading' => 'nullable|string|max:100',
            'port_of_discharge' => 'nullable|string|max:100',
            'etd' => 'nullable|date',
            'eta' => 'nullable|date|after:etd',
            'remarks' => 'nullable|string',
        ]);

        $validated['created_by'] = auth()->id();
        $shipment = Shipment::create($validated);

        return redirect()->route('admin.shipments.show', $shipment)
            ->with('success', 'Shipment created successfully.');
    }

    public function show(Shipment $shipment)
    {
        $shipment->load([
            'orderConfirmation', 
            'cartonDetails', 
            'bookingInvoice',
            'finalShipment',
            'creator'
        ]);
        
        return view('admin.shipment.shipments.show', compact('shipment'));
    }

    public function edit(Shipment $shipment)
    {
        $orders = OrderConfirmation::whereIn('status', ['confirmed', 'in_production', 'shipped'])
            ->get();
            
        return view('admin.shipment.shipments.edit', compact('shipment', 'orders'));
    }

    public function update(Request $request, Shipment $shipment)
    {
        $validated = $request->validate([
            'order_confirmation_id' => 'required|exists:order_confirmations,id',
            'shipment_date' => 'required|date',
            'vessel_flight_no' => 'nullable|string|max:100',
            'bill_of_lading' => 'nullable|string|max:100',
            'port_of_loading' => 'nullable|string|max:100',
            'port_of_discharge' => 'nullable|string|max:100',
            'etd' => 'nullable|date',
            'eta' => 'nullable|date|after:etd',
            'status' => 'required|in:draft,booked,in_transit,delivered,cancelled',
            'remarks' => 'nullable|string',
        ]);

        $shipment->update($validated);

        return redirect()->route('admin.shipments.show', $shipment)
            ->with('success', 'Shipment updated successfully.');
    }

    public function destroy(Shipment $shipment)
    {
        $shipment->delete();

        return redirect()->route('admin.shipments.index')
            ->with('success', 'Shipment deleted successfully.');
    }
}