<?php

namespace App\Http\Controllers;

use App\Models\FinalShipment;
use App\Models\Shipment;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;


class FinalShipmentController extends Controller
{
    public function index(Shipment $shipment)
    {
        $finalShipments = $shipment->finalShipment()->latest()->get();
        return view('admin.shipment.final-shipments.index', compact('shipment', 'finalShipments'));
    }

    public function create(Shipment $shipment)
    {
        $originalQuantity = $shipment->total_quantity;
        return view('admin.shipment.final-shipments.create', compact('shipment', 'originalQuantity'));
    }

    public function store(Request $request, Shipment $shipment)
    {
        $validated = $request->validate([
            'original_quantity' => 'required|integer|min:0',
            'adjusted_quantity' => 'required|integer',
            'adjustment_reason' => 'required|string|max:500',
            'final_shipment_date' => 'required|date',
            'inspection_remarks' => 'nullable|string',
        ]);

        $validated['shipment_id'] = $shipment->id;
        $validated['created_by'] = auth()->id();
        
        FinalShipment::create($validated);

        // Update shipment status if needed
        if ($shipment->status === 'in_transit') {
            $shipment->update(['status' => 'delivered']);
        }

        return redirect()->route('admin.shipments.final-shipments.index', $shipment)
            ->with('success', 'Final shipment details added successfully.');
    }

    public function show(Shipment $shipment, FinalShipment $finalShipment)
    {
        return view('admin.shipment.final-shipments.show', compact('shipment', 'finalShipment'));
    }

    public function edit(Shipment $shipment, FinalShipment $finalShipment)
    {
        return view('admin.shipment.final-shipments.edit', compact('shipment', 'finalShipment'));
    }

    public function update(Request $request, Shipment $shipment, FinalShipment $finalShipment)
    {
        $validated = $request->validate([
            'original_quantity' => 'required|integer|min:0',
            'adjusted_quantity' => 'required|integer',
            'adjustment_reason' => 'required|string|max:500',
            'final_shipment_date' => 'required|date',
            'status' => 'required|in:pending,adjusted,completed',
            'inspection_remarks' => 'nullable|string',
        ]);

        $finalShipment->update($validated);

        return redirect()->route('admin.shipments.final-shipments.index', $shipment)
            ->with('success', 'Final shipment details updated successfully.');
    }

    public function destroy(Shipment $shipment, FinalShipment $finalShipment)
    {
        $finalShipment->delete();

        return redirect()->route('admin.shipments.final-shipments.index', $shipment)
            ->with('success', 'Final shipment details deleted successfully.');
    }



  public function exportPackingList(Shipment $shipment, FinalShipment $finalShipment)
    {
        $finalShipment->load(['shipment.cartonDetails', 'shipment.orderConfirmation']);
        
        $pdf = PDF::loadView('admin.shipment.final-shipments.export.packing-list', compact('finalShipment', 'shipment'));
        
        return $pdf->download("packing-list-{$shipment->shipment_number}.pdf");
    }

    public function exportFinalInvoice(Shipment $shipment, FinalShipment $finalShipment)
    {
        $finalShipment->load(['shipment.orderConfirmation', 'creator']);
        
        $pdf = PDF::loadView('admin.shipment.final-shipments.export.final-invoice', compact('finalShipment', 'shipment'));
        
        return $pdf->download("final-invoice-{$shipment->shipment_number}.pdf");
    }



}