<?php

namespace App\Http\Controllers;

use App\Models\OrderConfirmation;
use App\Models\DocumentTemplate;
use App\Models\GeneratedDocument;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Storage;

class DocumentGeneratorController extends Controller
{
    public function index()
    {
        $orders = OrderConfirmation::whereIn('status', ['confirmed', 'in_production', 'shipped', 'delivered'])
            ->latest()
            ->get();
            
        $templates = DocumentTemplate::active()->get();
        
        return view('admin.document-generator.invoice-packing.index', compact('orders', 'templates'));
    }

    public function generateDocuments(Request $request)
    {
        $validated = $request->validate([
            'order_confirmation_id' => 'required|exists:order_confirmations,id',
            'document_types' => 'required|array',
            'document_types.*' => 'in:commercial_invoice,packing_list,certificate_origin,bill_of_lading,inspection_certificate,shipping_label',
            'template_id' => 'nullable|exists:document_templates,id',
            'include_signature' => 'boolean',
            'include_terms' => 'boolean',
        ]);

        $order = OrderConfirmation::findOrFail($validated['order_confirmation_id']);
        $generatedDocuments = [];

        foreach ($validated['document_types'] as $documentType) {
            $document = $this->generateSingleDocument($order, $documentType, $validated);
            if ($document) {
                $generatedDocuments[] = $document;
            }
        }

        return response()->json([
            'success' => true,
            'message' => count($generatedDocuments) . ' documents generated successfully',
            'documents' => $generatedDocuments
        ]);
    }

    private function generateSingleDocument($order, $documentType, $options)
    {
        try {
            // Pehle custom template check karein
            $customTemplate = null;
            if (isset($options['template_id']) && $options['template_id']) {
                $customTemplate = DocumentTemplate::find($options['template_id']);
            }

            // Agar custom template nahi hai to default template use karein
            if (!$customTemplate) {
                $customTemplate = DocumentTemplate::where('type', $documentType)
                    ->where('is_active', true)
                    ->where('is_default', true)
                    ->first();
            }

            $data = [
                'order' => $order,
                'document_type' => $documentType,
                'template' => $customTemplate,
                'include_signature' => $options['include_signature'] ?? false,
                'include_terms' => $options['include_terms'] ?? false,
                'generated_at' => now(),
            ];

            // Agar custom template hai to uske content ko use karein
            if ($customTemplate) {
                $htmlContent = $this->processTemplateContent($customTemplate->content, $order, $data);
                $pdf = PDF::loadHTML($htmlContent);
            } else {
                // Default template use karein
                $pdf = PDF::loadView('admin.document-generator.templates.' . $documentType, $data);
            }

            $pdf->setPaper('a4', 'portrait');

            $fileName = $documentType . '_' . $order->order_number . '_' . time() . '.pdf';
            $filePath = 'documents/' . $fileName;

            Storage::disk('public')->put($filePath, $pdf->output());

            $document = GeneratedDocument::create([
                'order_confirmation_id' => $order->id,
                'document_type' => $documentType,
                'file_name' => $fileName,
                'file_path' => $filePath,
                'file_size' => Storage::disk('public')->size($filePath),
                'generation_data' => $data,
                'generated_by' => auth()->id(),
            ]);

            return $document;

        } catch (\Exception $e) {
            \Log::error('Document generation failed: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Template content mein variables replace karein
     */
    private function processTemplateContent($content, $order, $data)
    {
        $variables = [
            'order_number' => $order->order_number ?? 'N/A',
            'customer_po_number' => $order->customer_po_number ?? 'N/A',
            'confirmed_quantity' => $order->confirmed_quantity ?? 'N/A',
            'confirmed_price' => number_format($order->confirmed_price, 2) ?? '0.00',
            'total_value' => number_format($order->confirmed_quantity * $order->confirmed_price, 2),
            'delivery_date' => $order->delivery_date ? $order->delivery_date->format('M d, Y') : 'N/A',
            'payment_terms' => $order->payment_terms ?? 'Standard payment terms apply',
            'special_instructions' => $order->special_instructions ?? '',
            'current_date' => now()->format('M d, Y'),
            'current_year' => now()->format('Y'),
        ];

        foreach ($variables as $key => $value) {
            $content = str_replace('{' . $key . '}', $value, $content);
            $content = str_replace('{{' . $key . '}}', $value, $content);
        }

        return $content;
    }

    public function previewDocument(Request $request)
    {
        $validated = $request->validate([
            'order_confirmation_id' => 'required|exists:order_confirmations,id',
            'document_type' => 'required|in:commercial_invoice,packing_list,certificate_origin,bill_of_lading,inspection_certificate,shipping_label',
        ]);

        $order = OrderConfirmation::findOrFail($validated['order_confirmation_id']);
        
        $data = [
            'order' => $order,
            'document_type' => $validated['document_type'],
            'preview' => true,
        ];

        $pdf = PDF::loadView('admin.document-generator.templates.' . $validated['document_type'], $data);
        
        return $pdf->stream('preview.pdf');
    }

    public function downloadDocument(GeneratedDocument $document)
    {
        if (!Storage::disk('public')->exists($document->file_path)) {
            return back()->with('error', 'Document file not found.');
        }

        return Storage::disk('public')->download($document->file_path, $document->file_name);
    }
}