<?php

namespace App\Http\Controllers;

use App\Models\CartonDetail;
use App\Models\Shipment;
use Illuminate\Http\Request;

class CartonDetailController extends Controller
{
    public function index(Shipment $shipment)
    {
        $cartons = $shipment->cartonDetails()->latest()->get();
        return view('admin.shipment.carton-details.index', compact('shipment', 'cartons'));
    }

    public function create(Shipment $shipment)
    {
        return view('admin.shipment.carton-details.create', compact('shipment'));
    }

    public function store(Request $request, Shipment $shipment)
    {
        $validated = $request->validate([
            'carton_number' => 'required|string|max:50',
            'quantity' => 'required|integer|min:1',
            'gross_weight' => 'nullable|numeric|min:0',
            'net_weight' => 'nullable|numeric|min:0',
            'volume' => 'nullable|numeric|min:0',
            'description' => 'nullable|string',
        ]);

        // Check if carton number already exists for this shipment
        $existingCarton = $shipment->cartonDetails()
            ->where('carton_number', $validated['carton_number'])
            ->first();

        if ($existingCarton) {
            return back()->withErrors(['carton_number' => 'Carton number already exists for this shipment.'])->withInput();
        }

        $validated['shipment_id'] = $shipment->id;
        CartonDetail::create($validated);

        return redirect()->route('admin.shipments.carton-details.index', $shipment)
            ->with('success', 'Carton detail added successfully.');
    }

    public function edit(Shipment $shipment, CartonDetail $cartonDetail)
    {
        return view('admin.shipment.carton-details.edit', compact('shipment', 'cartonDetail'));
    }

    public function update(Request $request, Shipment $shipment, CartonDetail $cartonDetail)
    {
        $validated = $request->validate([
            'carton_number' => 'required|string|max:50',
            'quantity' => 'required|integer|min:1',
            'gross_weight' => 'nullable|numeric|min:0',
            'net_weight' => 'nullable|numeric|min:0',
            'volume' => 'nullable|numeric|min:0',
            'description' => 'nullable|string',
        ]);

        // Check if carton number already exists for this shipment (excluding current carton)
        $existingCarton = $shipment->cartonDetails()
            ->where('carton_number', $validated['carton_number'])
            ->where('id', '!=', $cartonDetail->id)
            ->first();

        if ($existingCarton) {
            return back()->withErrors(['carton_number' => 'Carton number already exists for this shipment.'])->withInput();
        }

        $cartonDetail->update($validated);

        return redirect()->route('admin.shipments.carton-details.index', $shipment)
            ->with('success', 'Carton detail updated successfully.');
    }

    public function destroy(Shipment $shipment, CartonDetail $cartonDetail)
    {
        $cartonDetail->delete();

        return redirect()->route('admin.shipments.carton-details.index', $shipment)
            ->with('success', 'Carton detail deleted successfully.');
    }
}