<?php

namespace App\Http\Controllers;

use App\Models\BookingInvoice;
use App\Models\Shipment;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;


class BookingInvoiceController extends Controller
{
    public function index(Shipment $shipment)
    {
        $invoices = $shipment->bookingInvoice()->latest()->get();
        return view('admin.shipment.booking-invoices.index', compact('shipment', 'invoices'));
    }

    public function create(Shipment $shipment)
    {
        return view('admin.shipment.booking-invoices.create', compact('shipment'));
    }

     public function exportPdf(Shipment $shipment, BookingInvoice $bookingInvoice)
    {
        $bookingInvoice->load(['shipment.orderConfirmation', 'creator']);
        
        $pdf = PDF::loadView('admin.shipment.booking-invoices.export.pdf', compact('bookingInvoice', 'shipment'));
        
        return $pdf->download("booking-invoice-{$bookingInvoice->invoice_number}.pdf");
    }

    public function store(Request $request, Shipment $shipment)
    {
        $validated = $request->validate([
            'invoice_date' => 'required|date',
            'freight_charges' => 'required|numeric|min:0',
            'insurance_charges' => 'nullable|numeric|min:0',
            'handling_charges' => 'nullable|numeric|min:0',
            'other_charges' => 'nullable|numeric|min:0',
            'charges_description' => 'nullable|string',
        ]);

        $validated['shipment_id'] = $shipment->id;
        $validated['created_by'] = auth()->id();
        
        BookingInvoice::create($validated);

        return redirect()->route('admin.shipments.booking-invoices.index', $shipment)
            ->with('success', 'Booking invoice created successfully.');
    }

    public function show(Shipment $shipment, BookingInvoice $bookingInvoice)
    {
        return view('admin.shipment.booking-invoices.show', compact('shipment', 'bookingInvoice'));
    }

    public function edit(Shipment $shipment, BookingInvoice $bookingInvoice)
    {
        return view('admin.shipment.booking-invoices.edit', compact('shipment', 'bookingInvoice'));
    }

    public function update(Request $request, Shipment $shipment, BookingInvoice $bookingInvoice)
    {
        $validated = $request->validate([
            'invoice_date' => 'required|date',
            'freight_charges' => 'required|numeric|min:0',
            'insurance_charges' => 'nullable|numeric|min:0',
            'handling_charges' => 'nullable|numeric|min:0',
            'other_charges' => 'nullable|numeric|min:0',
            'charges_description' => 'nullable|string',
        ]);

        $bookingInvoice->update($validated);

        return redirect()->route('admin.shipments.booking-invoices.index', $shipment)
            ->with('success', 'Booking invoice updated successfully.');
    }

    public function destroy(Shipment $shipment, BookingInvoice $bookingInvoice)
    {
        $bookingInvoice->delete();

        return redirect()->route('admin.shipments.booking-invoices.index', $shipment)
            ->with('success', 'Booking invoice deleted successfully.');
    }
}