<?php

namespace App\Http\Controllers;

use App\Models\AdvancePayment;
use App\Models\Buyer;
use App\Models\OrderConfirmation;
use App\Models\PaymentLedger;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AdvancePaymentController extends Controller
{
    public function index()
    {
        $advancePayments = AdvancePayment::with(['buyer', 'orderConfirmation', 'creator'])
            ->latest()
            ->paginate(20);

        return view('admin.payment.advance-payments.index', compact('advancePayments'));
    }

    public function create()
    {
        $buyers = Buyer::all();
        $orders = OrderConfirmation::whereIn('status', ['confirmed', 'in_production'])->get();
        
        return view('admin.payment.advance-payments.create', compact('buyers', 'orders'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'buyer_id' => 'required|exists:buyers,id',
            'order_confirmation_id' => 'nullable|exists:order_confirmations,id',
            'amount' => 'required|numeric|min:0.01',
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:bank_transfer,lc,tt,cash,cheque',
            'transaction_id' => 'nullable|string|max:255',
            'remarks' => 'nullable|string',
        ]);

        DB::transaction(function () use ($validated) {
            $validated['created_by'] = auth()->id();
            $advancePayment = AdvancePayment::create($validated);

            // Update ledger
            $ledger = PaymentLedger::getOrCreateForBuyer($validated['buyer_id']);
            $ledger->updateBalance();
        });

        return redirect()->route('admin.advance-payments.index')
            ->with('success', 'Advance payment logged successfully.');
    }

    public function show(AdvancePayment $advancePayment)
    {
        $advancePayment->load(['buyer', 'orderConfirmation', 'allocations.factory', 'creator']);
        
        return view('admin.payment.advance-payments.show', compact('advancePayment'));
    }

    public function edit(AdvancePayment $advancePayment)
    {
        $buyers = Buyer::all();
        $orders = OrderConfirmation::whereIn('status', ['confirmed', 'in_production'])->get();
        
        return view('admin.payment.advance-payments.edit', compact('advancePayment', 'buyers', 'orders'));
    }

    public function update(Request $request, AdvancePayment $advancePayment)
    {
        $validated = $request->validate([
            'buyer_id' => 'required|exists:buyers,id',
            'order_confirmation_id' => 'nullable|exists:order_confirmations,id',
            'amount' => 'required|numeric|min:0.01',
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:bank_transfer,lc,tt,cash,cheque',
            'transaction_id' => 'nullable|string|max:255',
            'remarks' => 'nullable|string',
            'status' => 'required|in:pending,received,cancelled',
        ]);

        DB::transaction(function () use ($validated, $advancePayment) {
            $advancePayment->update($validated);

            // Update ledger
            $ledger = PaymentLedger::getOrCreateForBuyer($validated['buyer_id']);
            $ledger->updateBalance();
        });

        return redirect()->route('admin.advance-payments.index')
            ->with('success', 'Advance payment updated successfully.');
    }

    public function destroy(AdvancePayment $advancePayment)
    {
        DB::transaction(function () use ($advancePayment) {
            $buyerId = $advancePayment->buyer_id;
            $advancePayment->delete();

            // Update ledger
            $ledger = PaymentLedger::getOrCreateForBuyer($buyerId);
            $ledger->updateBalance();
        });

        return redirect()->route('admin.advance-payments.index')
            ->with('success', 'Advance payment deleted successfully.');
    }

    public function ledger()
    {
        $buyers = Buyer::has('advancePayments')->with(['ledger'])->get();
        
        return view('admin.payment.advance-payments.ledger', compact('buyers'));
    }

    public function buyerLedger(Buyer $buyer)
    {
        $ledger = PaymentLedger::getOrCreateForBuyer($buyer->id);
        $advancePayments = AdvancePayment::where('buyer_id', $buyer->id)
            ->with(['orderConfirmation', 'allocations'])
            ->latest()
            ->get();
            
        return view('admin.payment.advance-payments.buyer-ledger', compact('buyer', 'ledger', 'advancePayments'));
    }
}