<?php
// app/Http/Controllers/Admin/UserManagementController.php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use App\Models\Admin;
use App\Models\Role;
use App\Models\AuditLog;

class UserManagementController extends Controller
{
    public function index()
    {
        $users = Admin::with('roles')->where('id', '!=', auth('admin')->id())->get();
        $roles = Role::all();
        return view('admin.users.index', compact('users', 'roles'));
    }
    
    public function create()
    {
        $roles = Role::all();
        return view('admin.users.create', compact('roles'));
    }
    
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:admins',
            'password' => 'required|min:6',
            'roles' => 'required|array',
            'profile_image' => 'nullable|image|max:2048',
        ]);
        
        $user = Admin::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);
        
        $user->roles()->sync($request->roles);
        
        if ($request->hasFile('profile_image')) {
            $image = $request->file('profile_image');
            $name = time() . '.' . $image->getClientOriginalExtension();
            $image->move(public_path('uploads/admins'), $name);
            $user->profile_image = 'uploads/admins/' . $name;
            $user->save();
        }
        
        AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => 'created',
            'module' => 'Users',
            'description' => 'Created user: ' . $user->email,
            'ip_address' => $request->ip()
        ]);
        
        return redirect()->route('admin.users.index')->with('success', 'User created successfully');
    }
    
    public function edit(Admin $user)
    {
        if ($user->isSuperAdmin() && !auth('admin')->user()->isSuperAdmin()) {
            abort(403, 'Unauthorized action.');
        }
        
        $roles = Role::all();
        return view('admin.users.edit', compact('user', 'roles'));
    }
    
    public function update(Request $request, Admin $user)
    {
        if ($user->isSuperAdmin() && !auth('admin')->user()->isSuperAdmin()) {
            abort(403, 'Unauthorized action.');
        }
        
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:admins,email,' . $user->id,
            'password' => 'nullable|min:6',
            'roles' => 'required|array',
            'profile_image' => 'nullable|image|max:2048',
        ]);
        
        $user->name = $request->name;
        $user->email = $request->email;
        
        if ($request->filled('password')) {
            $user->password = Hash::make($request->password);
        }
        
        if (auth('admin')->user()->isSuperAdmin()) {
            $user->roles()->sync($request->roles);
        } else {
            $filteredRoles = Role::where('name', '!=', 'superadmin')
                ->whereIn('id', $request->roles)
                ->pluck('id');
            $user->roles()->sync($filteredRoles);
        }
        
        if ($request->hasFile('profile_image')) {
            $image = $request->file('profile_image');
            $name = time() . '.' . $image->getClientOriginalExtension();
            $image->move(public_path('uploads/admins'), $name);
            $user->profile_image = 'uploads/admins/' . $name;
        }
        
        $user->save();
        
        AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => 'updated',
            'module' => 'Users',
            'description' => 'Updated user: ' . $user->email,
            'ip_address' => $request->ip()
        ]);
        
        return redirect()->route('admin.users.index')->with('success', 'User updated successfully');
    }
    
    public function destroy(Admin $user)
    {
        if ($user->isSuperAdmin() || $user->id == auth('admin')->id()) {
            abort(403, 'Unauthorized action.');
        }
        
        $userEmail = $user->email;
        $user->delete();
        
        AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => 'deleted',
            'module' => 'Users',
            'description' => 'Deleted user: ' . $userEmail,
            'ip_address' => request()->ip()
        ]);
        
        return redirect()->route('admin.users.index')->with('success', 'User deleted successfully');
    }
}