<?php
// app/Http/Controllers/Admin/RoleManagementController.php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Role;
use App\Models\Permission;
use App\Models\AuditLog;

class RoleManagementController extends Controller
{
    public function index()
    {
        $roles = Role::withCount(['permissions', 'admins'])->get();
        return view('admin.roles.index', compact('roles'));
    }

    // 🔥 NEW: Create Role
    public function create()
    {
        return view('admin.roles.create');
    }

    // 🔥 NEW: Store Role
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:roles',
            'description' => 'required|string|max:500',
        ]);

        $role = Role::create([
            'name' => $request->name,
            'description' => $request->description,
            'is_default' => false,
        ]);

        AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => 'created',
            'module' => 'Roles',
            'description' => 'Created new role: ' . $role->name,
            'ip_address' => $request->ip()
        ]);

        return redirect()->route('admin.roles.index')->with('success', 'Role created successfully');
    }

    // 🔥 NEW: Edit Role
    public function edit(Role $role)
    {
        // Prevent editing superadmin role
        if ($role->name == 'superadmin' && !auth('admin')->user()->isSuperAdmin()) {
            abort(403, 'Unauthorized action.');
        }

        return view('admin.roles.edit', compact('role'));
    }

    // 🔥 NEW: Update Role
    public function update(Request $request, Role $role)
    {
        // Prevent editing superadmin role
        if ($role->name == 'superadmin' && !auth('admin')->user()->isSuperAdmin()) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'name' => 'required|string|max:255|unique:roles,name,' . $role->id,
            'description' => 'required|string|max:500',
        ]);

        $role->update([
            'name' => $request->name,
            'description' => $request->description,
        ]);

        AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => 'updated',
            'module' => 'Roles',
            'description' => 'Updated role: ' . $role->name,
            'ip_address' => $request->ip()
        ]);

        return redirect()->route('admin.roles.index')->with('success', 'Role updated successfully');
    }

    // 🔥 NEW: Delete Role
    public function destroy(Role $role)
    {
        // Prevent deleting superadmin and default roles
        if ($role->name == 'superadmin' || $role->is_default) {
            abort(403, 'Cannot delete system roles.');
        }

        // Check if role has users
        if ($role->admins()->count() > 0) {
            return redirect()->back()->with('error', 'Cannot delete role that has users assigned. Please reassign users first.');
        }

        $roleName = $role->name;
        $role->delete();

        AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => 'deleted',
            'module' => 'Roles',
            'description' => 'Deleted role: ' . $roleName,
            'ip_address' => request()->ip()
        ]);

        return redirect()->route('admin.roles.index')->with('success', 'Role deleted successfully');
    }

    public function editPermissions(Role $role)
    {
        if ($role->name == 'superadmin' && !auth('admin')->user()->isSuperAdmin()) {
            abort(403, 'Unauthorized action.');
        }

        $permissions = Permission::all()->groupBy('module');
        return view('admin.roles.permissions', compact('role', 'permissions'));
    }

    public function updatePermissions(Request $request, Role $role)
    {
        if ($role->name == 'superadmin' && !auth('admin')->user()->isSuperAdmin()) {
            abort(403, 'Unauthorized action.');
        }

        $role->permissions()->sync($request->permissions ?? []);

        AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => 'updated',
            'module' => 'Roles',
            'description' => 'Updated permissions for role: ' . $role->name,
            'ip_address' => $request->ip()
        ]);

        return redirect()->route('admin.roles.index')->with('success', 'Permissions updated successfully');
    }
}