<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\BlogCategory;
use App\Models\BlogPost;
use App\Models\BlogComment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class BlogController extends Controller
{
    public function index()
    {
        $posts = BlogPost::latest()->paginate(4);
        return view('admin.blogs.index', compact('posts'));
    }

    public function create()
    {
        $categories = BlogCategory::all();
        return view('admin.blogs.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'required|string|unique:blog_posts,slug',
            'short_description' => 'required|string',
            'content' => 'required|string',
            'tags' => 'required|string',
            'category_id' => 'required|exists:blog_categories,id',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048'
        ]);

        $post = new BlogPost;
        $post->title = $request->title;
        $post->slug = $request->slug;
        $post->tags = $request->tags;
        $post->short_description = $request->short_description;
        $post->content = $request->content;
        $post->category_id = $request->category_id;
            $post->author = auth('admin')->user()->name; // 🔥 dynamically grabbing logged-in user's name

if ($request->hasFile('image')) {
    $image = $request->file('image');
    $imageName = time().'_'.$image->getClientOriginalName();
    $image->move(public_path('blogs'), $imageName);
    $post->image = $imageName; // Save only the image name to DB
}

        $post->save();

        return redirect()->route('admin.blogs.index')->with('success', 'Blog post created successfully!');
    }

    public function show($slug)
    {
        $post = BlogPost::where('slug', $slug)->first();
        $comments = BlogComment::where('post_id', $post->id)->whereNull('parent_id')->get();

        return view('frontend.blogs.show', compact('post', 'comments'));
    }

    public function edit(BlogPost $post)
    {
        $categories = BlogCategory::all();
        return view('admin.blogs.edit', compact('post', 'categories'));
    }

    public function update(Request $request, BlogPost $post)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'required|string|unique:blog_posts,slug,' . $post->id,
            'short_description' => 'required|string',
            'content' => 'required|string',
            'tags' => 'required|string',
            'category_id' => 'required|exists:blog_categories,id',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048'
        ]);

        $post->title = $request->title;
        $post->slug = $request->slug;
        $post->tags = $request->tags;
        $post->short_description = $request->short_description;
        $post->content = $request->content;
        $post->category_id = $request->category_id;

        if ($request->hasFile('image')) {
            Storage::delete($post->image);
            $post->image = $request->file('image')->store('public/blogs');
        }

        $post->save();

        return redirect()->route('admin.posts.index')->with('success', 'Blog post updated successfully!');
    }

    public function destroy(BlogPost $post)
    {
        Storage::delete($post->image);
        $post->delete();

        return back()->with('success', 'Blog post deleted successfully!');
    }
}


